%   Copyright 2006-2010 Szymon Leski
%   s.leski@nencki.gov.pl

function iCSD2Dtool(varargin)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Initialization
smooth = 0;
color = 1;
redraw = 1;
handles = [];
time_frame=1;           % First frame to show
datasetdir = 'datasets';
data = [];
appdata.bgcolor = get(0,'DefaultUicontrolBackgroundColor');

%%% Choose initial data set or create new
if nargin==0
    fname = StartupFcn;
else
    fname = varargin{1};
end;
if isempty(fname)
    return;
end;

%%% Create main window
gui_layout;

%%% Load dataset
load_data;

%%% Start the main loop
timesdrcbk(handles.timesdr); % draw figure

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    function load_data
        % appdata.applybtn is true if Apply button has just been pressed
        % if yes, load new CSD data for the same voltage data
        % if no, load voltage data first
        if ~appdata.applybtn
            disp(['loading ' fullfile(datasetdir, [fname '-pots.mat'])]);
            data = load(fullfile(datasetdir, [fname '-pots.mat']));
            set(handles.climssdr, 'Value', 1);
            appdata.climsfactor = 1;
            set(handles.climstxt, 'String', '100%')
        end;
        switch appdata.method
            case 'traditional'
                method = 'trad';
            case 'step'
                method = 'step';
            case 'linear'
                method = 'lin';
            case 'spline'
                method = 'spline';
        end
        switch appdata.boundary
            case 'B boundary'
                bdry = 'B';
            case 'D boundary'
                bdry = 'D';
            otherwise
                bdry = appdata.boundary;
        end;
        data.csd2 = Get_csd(fname, method, bdry, appdata.h);

        data.clims = [-1 1]*max(abs(data.csd2(:)))/6;
        data.climspot = [-1 1]*max(abs(data.pots(:)))/4;
        if data.clims(1) == data.clims(2)
            data.clims = [-1 1];
        end;
        if data.climspot(1) == data.climspot(2)
            data.climspot = [-1 1];
        end;
        data.nt = size(data.csd2,1);
        data.nnx = size(data.csd2,2);
        data.nny = size(data.csd2,3);
        data.dimx = (size(data.pots, 2)-1)*data.dx;
        data.dimy = (size(data.pots, 3)-1)*data.dy;
        data.tvec = (0:data.nt-1)*data.dt + data.t0;
        if ~appdata.applybtn
            set(handles.timesdr, ...
                'Min',          data.t0, ...
                'Max',          data.t0 + (data.nt - 1) * data.dt, ...
                'Value',        data.t0, ...
                'SliderStep',   1/(data.nt - 1)*[1 10]);
            set(handles.timeedt, ...
                'String',       num2str(get(handles.timesdr, 'Value')));
        end;
        set(handles.ntedt, 'String', num2str(data.nt));
        set(handles.nxedt, 'String', num2str(data.nx));
        set(handles.nyedt, 'String', num2str(data.ny));
        set(handles.dtedt, 'String', num2str(data.dt));
        set(handles.t0edt, 'String', num2str(data.t0));
        set(handles.dxedt, 'String', num2str(data.dx));
        set(handles.dyedt, 'String', num2str(data.dy));
        set(handles.nametxt, 'String', fname);
        set(handles.comtxt, 'String', data.comment);
        redraw = 1;
    end % of function

    function draw_figure(fig_h, ax1_h, ax2_h, climsax_h)
        if nargin == 0
            fig_h = handles.fig1;
            ax1_h = handles.axes1;
            ax2_h = handles.axes2;
            climsax_h = handles.climsax;
        end;
        set(fig_h, 'CurrentAxes', ax2_h)
        if redraw
            handles.img2 = ...
                imagesc(linspace(0, data.dimx, data.nnx), ...
                linspace(0, data.dimy, data.nny), ...
                squeeze(data.csd2(time_frame,:,:))', ...
                data.clims*appdata.climsfactor);
        else
            set(handles.img2, 'CData', ...
                squeeze(data.csd2(time_frame,:,:))');
            set(ax2_h, 'Clim', data.clims*appdata.climsfactor);
        end;


        if color
            colormap(flipud(jet(1024)));
        else
            colormap(gray(1024));
        end;
        axis image

        set(ax2_h, ...
            'Xtick',    0:data.dx:data.dimx, ...
            'Ytick',    0:data.dy:data.dimy);
        mark_missing
        set(fig_h, 'CurrentAxes', ax1_h)
        if smooth
            if redraw
                handles.img1 = ...
                    imagesc(linspace(0, data.dimx, data.nnx), ...
                    linspace(0, data.dimy, data.nny), ...
                    squeeze(data.pots2(time_frame,:,:))', ...
                    data.climspot*appdata.climsfactor);
            else
                set(handles.img1, 'CData', ...
                    squeeze(data.pots2(time_frame,:,:))');
                set(ax1_h, 'Clim', data.climspot*appdata.climsfactor);
            end;
        else
            if redraw
                handles.img1 = ...
                    imagesc(linspace(0, data.dimx, data.nnx), ...
                    linspace(0, data.dimy, data.nny), ...
                    squeeze(data.pots(time_frame,:,:))', ...
                    data.climspot*appdata.climsfactor);
            else
                set(handles.img1, 'CData', ...
                    squeeze(data.pots(time_frame,:,:))');
                set(ax1_h, 'Clim', data.climspot*appdata.climsfactor);
            end;
        end;
        if color
            colormap(flipud(jet(1024)));
        else
            colormap(gray(1024));
        end;
        %title('Potentials')
        axis image
        set(ax1_h, ...
            'Xtick',    0:data.dx:data.dimx, ...
            'Ytick',    0:data.dy:data.dimy);
        mark_missing
        if redraw
            set(fig_h, 'CurrentAxes', climsax_h);
            imagesc(linspace(1,-1,size(colormap,1))');
            axis off
        end;
        if nargin == 0
            oldpos = get(handles.fig1, 'Position');
            drawnow;
            set(handles.fig1, 'Position', oldpos)
        else
            drawnow;
        end;
        redraw = 0;
    end

    function mark_missing
        hold on
        plot((data.msx-1)*data.dx, (data.msy-1)*data.dy,  'x')
        hold off
    end % of function

    function applybtncbk(src, evnt) %#ok<INUSD>
        appdata.applybtn = 1;
        set(handles.applybtn, 'Enable', 'off', 'String', 'Wait...');
        drawnow
        load_data;
        appdata.applybtn = 0;
        set(handles.applybtn, 'Enable', 'on', 'String', 'Apply');
        timesdrcbk(handles.timesdr);
    end % of callback

    function backbtncbk(src, evnt) %#ok<INUSD>
        settimesdrval(get(handles.timesdr, 'Value') - data.dt);
    end % of callback

    function fwdbtncbk(src, evnt) %#ok<INUSD>
        settimesdrval(get(handles.timesdr, 'Value') + data.dt);
    end % of callback

    function smoothbtncbk(src, evnt) %#ok<INUSD>
        smooth = 1 - smooth;
        draw_figure;
    end % of callback

    function colorbtncbk(src, evnt) %#ok<INUSD>
        color = 1 - color;
        set(src, 'String', appdata.colorstring{color+1});
        draw_figure;
    end % of callback

    function timesdrcbk(src, evnt) %#ok<INUSD>
        slider_value = get(src, 'Value');
        time_frame = find(data.tvec >= slider_value, 1);
        set(handles.timeedt, ...
            'String', num2str(data.tvec(time_frame), '%3.1f'));
        draw_figure;
    end % of callback

    function mtdpopcbk(src, evnt) %#ok<INUSD>
        appdata.method = appdata.mtdpopstr{get(src, 'Value')};
        if strcmp(appdata.method, 'traditional')
            appdata.bdrpopstr = appdata.bdrtradstr;
            set(handles.htxt, 'Enable', 'off');
            set(handles.htxt2, 'Enable', 'off');
            set(handles.hedt, 'Enable', 'off');
            set(handles.bdrpop, 'String', appdata.bdrpopstr);
            set(handles.bdrpop, 'Value', appdata.bdrtrad)
        else
            appdata.bdrpopstr = appdata.bdricsdstr;
            set(handles.htxt, 'Enable', 'on');
            set(handles.htxt2, 'Enable', 'on');
            set(handles.hedt, 'Enable', 'on');
            set(handles.bdrpop, 'String', appdata.bdrpopstr);
            set(handles.bdrpop, 'Value', appdata.bdricsd)
        end;
        appdata.boundary = ...
            appdata.bdrpopstr{get(handles.bdrpop, 'Value')};
    end % of callback

    function bdrpopcbk(src, evnt) %#ok<INUSD>
        appdata.boundary = appdata.bdrpopstr{get(src, 'Value')};
        if strcmp(appdata.method, 'traditional')
            appdata.bdrtrad = get(src, 'Value');
        else
            appdata.bdricsd = get(src, 'Value');
        end;
    end % of callback

    function timeedtcbk(src, evnt) %#ok<INUSD>
        edit_value = str2double(get(src, 'String'));
        if isnumeric(edit_value) && length(edit_value) == 1
            settimesdrval(edit_value);
        else
            settimesdrval(get(handles.timesdr, 'Value'));
        end;
    end % of callback

    function climssdrcbk(src, evnt) %#ok<INUSD>
        val = get(src, 'Value');
        appdata.climsfactor = val;
        set(handles.climstxt, 'String', [num2str(round(val*100)) '%']);
        draw_figure;
    end % of callback

    function hedtcbk(src, evnt) %#ok<INUSD>
        edit_value = str2double(get(src, 'String'));
        if isnumeric(edit_value) && length(edit_value) == 1 ...
                && edit_value > 0
            appdata.h = edit_value;
        else
            set(src, 'String', num2str(appdata.h));
        end;
    end % of callback

    function settimesdrval(val)
        smin = get(handles.timesdr, 'Min');
        smax = get(handles.timesdr, 'Max');
        if val>smax
            val = smax;
        end;
        if val<smin
            val = smin;
        end;
        set(handles.timesdr, 'Value', val);
        timesdrcbk(handles.timesdr);
    end % of function

    function FileNewcbk(src, evnt) %#ok<INUSD>
        newfname = NewDatasetGui;
        if ~isempty(newfname) && ~strcmp(fname, newfname)
            button = questdlg(...
                {['You have created new data set (' ...
                newfname ').'], ...
                'Do you want to load it?'}, ...
                'New data set', 'Yes', 'No', 'Yes');
            if strcmp(button, 'Yes')
                fname = newfname;
                load_data;
                time_frame = 1;
                timesdrcbk(handles.timesdr);
            end;
        end;


    end % of callback

    function FileOpencbk(src, evnt) %#ok<INUSD>
        files = dir(fullfile(datasetdir, '*-pots.mat'));
        datasets = {files.name};
        for ii = 1:length(datasets)
            datasets{ii} = datasets{ii}(1:end-9);
        end;
        [sel, ok] = listdlg(...
            'ListString',       datasets, ...
            'SelectionMode',    'single', ...
            'Name',             'Open...', ...
            'PromptString',     'Select dataset');
        if ok
            if ~strcmp(fname, datasets{sel})
                fname = datasets{sel};
                load_data;
                time_frame = 1;
                timesdrcbk(handles.timesdr);
            end;
        end;
    end % of callback

    function fname = StartupFcn
        fname = '';
        files = dir(fullfile(datasetdir, '*-pots.mat'));
        datasets = {files.name};
        for ii = 1:length(datasets)
            datasets{ii} = datasets{ii}(1:end-9);
        end;
        handles.fig0 = figure(...
            'Position',     [ 400 300 420 400],  ...
            'Name',         'iCSD 2D', ...
            'NumberTitle',  'off', ...
            'Renderer',     'zbuffer', ...
            'MenuBar',      'none', ...
            'resize',       'off', ...
            'Color',        appdata.bgcolor);
        movegui(handles.fig0, 'onscreen');
        handles.f0dstlist = uicontrol(...
            'Position',     [20 20 120 200], ...
            'Style',        'listbox', ...
            'Min',          0, ...
            'Max',          0, ...
            'String',       datasets, ...
            'Callback',     @f0dstlistcbk);
        handles.f0opnbtn = uicontrol(...
            'Position',     [240, 20, 80, 40], ...
            'Style',        'pushbutton', ...
            'String',       'Open', ...
            'Callback',     @f0opncbk);
        handles.f0opntit = uicontrol(...
            'Position',     [20, 220, 380, 30], ...
            'Style',        'text', ...
            'String',       'Open existing data set', ...
            'FontWeight',   'bold');
        handles.f0opntxt = uicontrol(...
            'Position',     [160, 80, 240, 120], ...
            'Style',        'text', ...
            'String',       '');
        
        handles.f0newbtn = uicontrol(...
            'Position',     [170, 300, 80, 40], ...
            'Style',        'pushbutton', ...
            'String',       'New', ...
            'Callback',     @f0newcbk);
        handles.f0newtit = uicontrol(...
            'Position',     [20, 340, 380, 30], ...
            'Style',        'text', ...
            'String',       'Create new data set', ...
            'FontWeight',   'bold');

        f0dstlistcbk(handles.f0dstlist, []);
        uiwait(handles.fig0);

        function f0opncbk(src, event)  %#ok<INUSD>
            fname = datasets{get(handles.f0dstlist, 'Value')};
            close(handles.fig0)
        end % of callback
        
        function f0newcbk(src, event)  %#ok<INUSD>
            fname = NewDatasetGui;
            close(handles.fig0)
        end % of callback
        
        function f0dstlistcbk(src, event)  %#ok<INUSD>
            clicktype = get(handles.fig0, 'SelectionType');
            switch clicktype
                case 'open'
                    fname = datasets{get(src, 'Value')};
                    close(handles.fig0)
                case 'normal'
                    dstinf = load(fullfile(datasetdir, ...
                        [datasets{get(src, 'Value')} '-pots.mat']), ...
                        'nx', 'ny', 'dx', 'dy', 'dt', 't0', 'pots', ...
                        'comment');
                    dstinf.nt = num2str(size(dstinf.pots, 1));
                    set(handles.f0opntxt, 'String', { ...
                        datasets{get(src, 'Value')},  ...
                        dstinf.comment, '',  ... 
                        ['nx: ' num2str(dstinf.nx) ...
                        ' ny: ' num2str(dstinf.ny) ],  ... 
                        ['nt: ' dstinf.nt], ... 
                        ['dx: ' num2str(dstinf.dx) ...
                        ' dy: ' num2str(dstinf.dy) ],  ... 
                        ['t0: ' num2str(dstinf.t0) ...
                        ' dt: ' num2str(dstinf.dt) ] ... 
                        });
            end;
                    
        end % of callback
        
    end % of function

    function FileQuitcbk(src, evnt) %#ok<INUSD>
        close(handles.fig1)
    end % of callback

    function HelpAboutcbk(src, evnt) %#ok<INUSD>
        helpdlg(appdata.helpstring, appdata.helpdlgname);
    end % of callback

    function savebtncbk(src, evnt) %#ok<INUSD>
        newfig = figure('Renderer',     'zbuffer');
        fig_fname = fullfile('figures', [fname '_t=' ...
            num2str(data.tvec(time_frame)) '.png']);
        ax1 = axes('Position', [0.05 0.1 0.375 0.7] );
        ax2 = axes('Position', [0.475 0.1 0.375 0.7] );
        cax = axes('Position', [0.9 0.1 0.05 0.7] );
        redraw = 1;
        draw_figure(newfig, ax1, ax2, cax);
        redraw = 1;
        title(ax1, 'Potentials')
        title(ax2, ['CSD, ', appdata.method ', ' appdata.boundary]);
        print(fig_fname, '-dpng',['-f' num2str(newfig)])
        set(newfig, 'Name', ['Saved as: ' fig_fname], 'NumberTitle', 'off');
    end % of callback

    function gui_layout
        %%% GUI layout for iCSDtool
        handles.fig1 = figure(...
            'Position',     [ 300 200 800 680],  ...
            'Renderer',     'zbuffer', ...
            'MenuBar',      'none', ...
            'Name',         'iCSD 2D', ...
            'NumberTitle',  'off', ...
            'resize',       'off', ...
            'Color',        appdata.bgcolor);
        movegui(handles.fig1, 'onscreen');
        handles.File = uimenu(...
            'Label',        'File');
        handles.FileNew = uimenu(handles.File, ...
            'Label',        'New...', ...
            'Callback',     @FileNewcbk, ...
            'Separator',    'off');
        handles.FileOpen = uimenu(handles.File, ...
            'Label',        'Open...', ...
            'Callback',     @FileOpencbk, ...
            'Separator',    'off');
        handles.FileQuit = uimenu(handles.File, ...
            'Label',        'Quit', ...
            'Callback',     @FileQuitcbk, ...
            'Separator',    'on');
        handles.Help = uimenu(...
            'Label',        'Help');
        handles.HelpAbout = uimenu(handles.Help, ...
            'Label',        'About', ...
            'Callback',     @HelpAboutcbk, ...
            'Separator',    'off');
        appdata.helpstring = ...
            '2D CSD tool. (C) Szymon Leski 2008-2010';
        appdata.helpdlgname = ...
            'About';

        handles.uipanel1 = uipanel(...
            'Title',       'Data set', ...
            'Units',       'pixels', ...
            'Position',     [60 480 300 180]);
        handles.nametxt = uicontrol(...
            'Style',        'text', ...
            'String',       'Dataset: ', ...
            'HorizontalAlignment', 'left', ...
            'Position',     [20 130 140 20], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.comtxt = uicontrol(...
            'Style',        'text', ...
            'String',       'Description', ...
            'HorizontalAlignment', 'left', ...
            'Position',     [20 100 260 20], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.nttxt = uicontrol(...
            'Style',        'text', ...
            'String',       'nt:', ...
            'Position',     [20 40 30 20], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.ntedt = uicontrol(...
            'Style',        'text', ...
            'String',       '--', ...
            'Position',     get(handles.nttxt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.nxtxt = uicontrol(...
            'Style',        'text', ...
            'String',       'nx:', ...
            'Position',     get(handles.ntedt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.nxedt = uicontrol(...
            'Style',        'text', ...
            'String',       '--', ...
            'Position',     get(handles.nxtxt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.nytxt = uicontrol(...
            'Style',        'text', ...
            'String',       'ny:', ...
            'Position',     get(handles.nxedt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.nyedt = uicontrol(...
            'Style',        'text', ...
            'String',       '--', ...
            'Position',     get(handles.nytxt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);

        handles.dxtxt = uicontrol(...
            'Style',        'text', ...
            'String',       'dx:', ...
            'Position',     [20 10 30 20], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.dxedt = uicontrol(...
            'Style',        'text', ...
            'String',       '1', ...
            'Position',     get(handles.dxtxt, 'Position') + [30 0 0 0], ...
            'Callback',     @numedtcbk, ...
            'Parent',       handles.uipanel1);
        handles.dytxt = uicontrol(...
            'Style',        'text', ...
            'String',       'dy:', ...
            'Position',     get(handles.dxedt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.dyedt = uicontrol(...
            'Style',        'text', ...
            'String',       '1', ...
            'Position',     get(handles.dytxt, 'Position') + [30 0 0 0], ...
            'Callback',     @numedtcbk, ...
            'Parent',       handles.uipanel1);
        handles.t0txt = uicontrol(...
            'Style',        'text', ...
            'String',       't0:', ...
            'Position',     get(handles.dyedt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.t0edt = uicontrol(...
            'Style',        'text', ...
            'String',       '0', ...
            'Position',     get(handles.t0txt, 'Position') + [30 0 0 0], ...
            'Callback',     @numedtcbk, ...
            'Parent',       handles.uipanel1);
        handles.dttxt = uicontrol(...
            'Style',        'text', ...
            'String',       'dt:', ...
            'Position',     get(handles.t0edt, 'Position') + [30 0 0 0], ...
            'Callback',     '', ...
            'Parent',       handles.uipanel1);
        handles.dtedt = uicontrol(...
            'Style',        'text', ...
            'String',       '1', ...
            'Position',     get(handles.dttxt, 'Position') + [30 0 0 0], ...
            'Callback',     @numedtcbk, ...
            'Parent',       handles.uipanel1);


        handles.uipanel2 = uipanel(...
            'Title',       'Current source density method', ...
            'Units',       'pixels', ...
            'Position',     [400 480 300 180]);
        handles.applybtn = uicontrol(...
            'Style',        'pushbutton', ...
            'String',       'Apply', ...
            'Position',     [200 20 80 40], ...
            'Callback',     @applybtncbk, ...
            'Parent',       handles.uipanel2, ...
            'TooltipString', 'Apply current CSD method');
        appdata.applybtn = 0;
        appdata.mtdpopstr = {...
            'traditional', 'step', 'linear', 'spline'};
        appdata.bdrtradstr = {...
            'no', 'Vaknin'};
        appdata.bdricsdstr = {...
            'no', 'B boundary', 'D boundary'};
        appdata.bdrpopstr = appdata.bdrtradstr;
        appdata.bdrtrad = 1; % remember values when switching
        appdata.bdricsd = 1; % between trad and icsd
        handles.mtdtxt = uicontrol(...
            'Style',        'text', ...
            'String',       'Method: ', ...
            'HorizontalAlignment', 'left', ...
            'Position',     [20 130 80 20], ...
            'Parent',       handles.uipanel2);
        handles.mtdpop = uicontrol(...
            'Style',        'popupmenu', ...
            'String',       appdata.mtdpopstr, ...
            'Position',     [140 130 140 20], ...
            'Callback',     @mtdpopcbk, ...
            'Parent',       handles.uipanel2);
        handles.bdrtxt = uicontrol(...
            'Style',        'text', ...
            'String',       'Boundary: ', ...
            'HorizontalAlignment', 'left', ...
            'Position',     [20 90 80 20], ...
            'Parent',       handles.uipanel2);
        handles.bdrpop = uicontrol(...
            'Style',        'popupmenu', ...
            'String',       appdata.bdrpopstr, ...
            'Position',     [140 90 140 20], ...
            'Callback',     @bdrpopcbk, ...
            'Parent',       handles.uipanel2);
        appdata.method = appdata.mtdpopstr{...
            get(handles.mtdpop, 'Value')};
        appdata.boundary = appdata.bdrpopstr{...
            get(handles.bdrpop, 'Value')};
        handles.htxt = uicontrol(...
            'Style',        'text', ...
            'String',       'h = ', ...
            'Position',     [20 30 40 20], ...
            'Enable',       'off', ...
            'Parent',       handles.uipanel2);
        handles.hedt = uicontrol(...
            'Style',        'edit', ...
            'Position',     [60 30 40 20], ...
            'String',       '1', ...
            'Callback',     @hedtcbk, ...
            'Enable',       'off', ...
            'Parent',       handles.uipanel2);
        appdata.h = str2double(get(handles.hedt, 'String'));
        handles.htxt2 = uicontrol(...
            'Style',        'text', ...
            'String',       '[mm]', ...
            'Position',     [100 30 40 20], ...
            'Enable',       'off', ...
            'Parent',       handles.uipanel2);

        handles.axes1 = axes(...
            'Units',       'pixels', ...
            'Position',     [60 130 300 300]);
        handles.ax1txt= uicontrol(...
            'Style',        'text', ...
            'String',       'Potentials', ...
            'FontWeight',   'bold', ...
            'Position',     [170 440 80 20]);
        handles.axes2 = axes(...
            'Units',       'pixels', ...
            'Position',     [400 130 300 300]);
        handles.ax1txt= uicontrol(...
            'Style',        'text', ...
            'String',       'CSD', ...
            'FontWeight',   'bold', ...
            'Position',     [510 440 80 20]);
        handles.climsax = axes(...
            'Units',       'pixels', ...
            'Position',     [720 130 20 300]);
        handles.climssdr = uicontrol(...
            'Style',        'slider', ...
            'Position',     [760 130 20 300], ...
            'Callback',     @climssdrcbk, ...
            'Min',          0.01, ...
            'Max',          4, ...
            'Value',        1);
        appdata.climsfactor = get(handles.climssdr, 'Value');
        handles.climstxt = uicontrol(...
            'Style',        'text', ...
            'Position',     [750, 100, 40, 20], ...
            'String',       [num2str(round(appdata.climsfactor*100)) '%']);
        handles.sinktxt = uicontrol(...
            'Style',        'text', ...
            'Position',     [710, 100, 40, 20], ...
            'String',       'Sink');
        handles.sourcetxt = uicontrol(...
            'Style',        'text', ...
            'Position',     [705, 440, 50, 20], ...
            'String',       'Source');
        
        handles.backbtn = uicontrol(...
            'Style',        'pushbutton', ...
            'String',       'Back', ...
            'Position',     [60 20 60 40], ...
            'Callback',     @backbtncbk);
        handles.fwdbtn = uicontrol(...
            'Style',        'pushbutton', ...
            'String',       'Fwd', ...
            'Position',     [140 20 60 40], ...
            'Callback',     @fwdbtncbk);
        handles.savebtn = uicontrol(...
            'Style',        'pushbutton', ...
            'String',       'Save PNG', ...
            'Position',     [620 20 80 40], ...
            'Callback',     @savebtncbk);
        handles.colorbtn = uicontrol(...
            'Style',        'togglebutton', ...
            'String',       'Color', ...
            'Position',     [510 20 80 40], ...
            'Callback',     @colorbtncbk);
        appdata.colorstring = {'Grayscale', 'Color'};
        handles.smoothbtn = uicontrol(...
            'Style',        'togglebutton', ...
            'String',       'Smooth V', ...
            'Position',     [400 20 80 40], ...
            'Callback',     @smoothbtncbk);

        handles.timesdr = uicontrol(...
            'Style',        'slider', ...
            'Position',     [60 80 640 20], ...
            'Callback',     @timesdrcbk);
        handles.timetxt = uicontrol(...
            'Style',        'text', ...
            'String',       'Time:', ...
            'Position',     [220 30 52 20], ...
            'Callback',     '');
        handles.timeedt = uicontrol(...
            'Style',        'edit', ...
            'Position',     [280 30 80 20], ...
            'Callback',     @timeedtcbk);
    end % of gui layout

end % of all
